/// @file udev.h
///
/// @brief Describes the @ref Udev class
///
/// @description The udev class aims to interface with the udev library,
/// gathering information from devices and monitoring events.
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see EventHandler DeviceDetector DeviceInfo

#ifndef __DDETECT_UDEV__
#define __DDETECT_UDEV__

#include <libudev.h>

#include "deviceDetector.h"

namespace adit {
namespace uspi {

/// @brief The udev class wrapper singleton
///
/// Object created out of the udev interface. Used to monitor for specific
/// devices. Once new device are found, they are added to the device list and
/// the DeviceDetector is used to check for client notification.
/// This object is to be initialized and used by only one DeviceDetector object.
class Udev {
    struct udev *mUdev;             ///< Udev instance
    struct udev_monitor *mMon;      ///< Udev monitor instance

    int mFd;                        ///< fd File descriptor for events
    DeviceDetector *mDevDetector;   ///< The DeviceDetector reference

    Udev();
    ~Udev();

    /// Check each parent of a specific subsystem ("usb" or "bluetooth").
    void checkParent(struct udev_enumerate *, const char *);

public:
    /// The instance provider
    static Udev& getInstance()
    {
        static Udev u;
        return u;
    };

    /// Put the object in standalone mode
    void setStandAlone(void);
    /// Dispatch events received on the @ref mFd file descriptor.
    int dispatchEvent();

    /// Check the complete list of devices for notification.
    void checkAll(void);

    /// Handle a udev device event.
    void handleDevice(struct udev_device *, std::string&, bool check = true);

    /// Returns the udev file descriptor
    int getFd() { return mFd; };
    /// Sets the DeviceDetector reference
    void setDeviceDetector(DeviceDetector *devDetector) {
        mDevDetector = devDetector;
    };

    /// Receive device from the udev monitor.
    struct udev_device *monitorReceiveDevice(void);
};

} // namespace uspi
} // namespace adit
#endif
